#![cfg(aes)]

use wolfssl::wolfcrypt::aes::*;

const BIG_MSG: [u8; 384] = [
    0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
    0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
    0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
    0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20,
    0x61,0x20,0x64,0x75,0x6c,0x6c,0x20,0x62,
    0x6f,0x79,0x2e,0x20,0x41,0x6c,0x6c,0x20,
    0x77,0x6f,0x72,0x6b,0x20,0x61,0x6e,0x64,
    0x20,0x6e,0x6f,0x20,0x70,0x6c,0x61,0x79,
    0x20,0x6d,0x61,0x6b,0x65,0x73,0x20,0x4a,
    0x61,0x63,0x6b,0x20,0x61,0x20,0x64,0x75,
    0x6c,0x6c,0x20,0x62,0x6f,0x79,0x2e,0x20,
    0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
    0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
    0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
    0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20,
    0x61,0x20,0x64,0x75,0x6c,0x6c,0x20,0x62,
    0x6f,0x79,0x2e,0x20,0x41,0x6c,0x6c,0x20,
    0x77,0x6f,0x72,0x6b,0x20,0x61,0x6e,0x64,
    0x20,0x6e,0x6f,0x20,0x70,0x6c,0x61,0x79,
    0x20,0x6d,0x61,0x6b,0x65,0x73,0x20,0x4a,
    0x61,0x63,0x6b,0x20,0x61,0x20,0x64,0x75,
    0x6c,0x6c,0x20,0x62,0x6f,0x79,0x2e,0x20,
    0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
    0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
    0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
    0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20,
    0x61,0x20,0x64,0x75,0x6c,0x6c,0x20,0x62,
    0x6f,0x79,0x2e,0x20,0x41,0x6c,0x6c,0x20,
    0x77,0x6f,0x72,0x6b,0x20,0x61,0x6e,0x64,
    0x20,0x6e,0x6f,0x20,0x70,0x6c,0x61,0x79,
    0x20,0x6d,0x61,0x6b,0x65,0x73,0x20,0x4a,
    0x61,0x63,0x6b,0x20,0x61,0x20,0x64,0x75,
    0x6c,0x6c,0x20,0x62,0x6f,0x79,0x2e,0x20,
    0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
    0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
    0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
    0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20,
    0x61,0x20,0x64,0x75,0x6c,0x6c,0x20,0x62,
    0x6f,0x79,0x2e,0x20,0x41,0x6c,0x6c,0x20,
    0x77,0x6f,0x72,0x6b,0x20,0x61,0x6e,0x64,
    0x20,0x6e,0x6f,0x20,0x70,0x6c,0x61,0x79,
    0x20,0x6d,0x61,0x6b,0x65,0x73,0x20,0x4a,
    0x61,0x63,0x6b,0x20,0x61,0x20,0x64,0x75,
    0x6c,0x6c,0x20,0x62,0x6f,0x79,0x2e,0x20,
    0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
    0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
    0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
    0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20
];

#[test]
#[cfg(aes_cbc)]
fn test_cbc_encrypt_decrypt() {
    let mut cbc = CBC::new().expect("Failed to create CBC");
    let key: &[u8; 16] = b"0123456789abcdef";
    let iv: &[u8; 16] = b"1234567890abcdef";
    let msg: [u8; 16] = [
        0x6e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74,
        0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20,
    ];
    let expected_cipher: [u8; 16] = [
        0x95, 0x94, 0x92, 0x57, 0x5f, 0x42, 0x81, 0x53,
        0x2c, 0xcc, 0x9d, 0x46, 0x77, 0xa2, 0x33, 0xcb
    ];
    cbc.init_encrypt(key, iv).expect("Error with init_encrypt()");
    let mut cipher: [u8; 16] = [0; 16];
    cbc.encrypt(&msg, &mut cipher).expect("Error with encrypt()");
    assert_eq!(&cipher, &expected_cipher);
    let mut plain_out = [0; 16];
    cbc.init_decrypt(key, iv).expect("Error with init_decrypt()");
    cbc.decrypt(&cipher, &mut plain_out).expect("Error with decrypt()");
    assert_eq!(&plain_out, &msg);
}

#[test]
#[cfg(aes_cbc)]
fn test_cbc_big_msg() {
    let mut cbc = CBC::new().expect("Failed to create CBC");
    let big_key = b"0123456789abcdeffedcba9876543210";
    let iv: &[u8; 16] = b"1234567890abcdef";
    let mut big_cipher: [u8; 384] = [0; 384];
    let mut big_plain: [u8; 384] = [0; 384];
    cbc.init_encrypt(big_key, iv).expect("Error with init_encrypt()");
    for i in (0..384).step_by(16) {
        cbc.encrypt(&BIG_MSG[i..(i + 16)], &mut big_cipher[i..(i + 16)]).expect("Error with encrypt()");
    }
    cbc.init_decrypt(big_key, iv).expect("Error with init_decrypt()");
    for i in (0..384).step_by(16) {
        cbc.decrypt(&big_cipher[i..(i + 16)], &mut big_plain[i..(i + 16)]).expect("Error with decrypt()");
    }
    assert_eq!(big_plain, BIG_MSG);
}

#[test]
#[cfg(aes_ccm)]
fn test_ccm_encrypt_decrypt() {
    let key: [u8; 16] = [
        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
        0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf
    ];
    let nonce: [u8; 13] = [
        0x00, 0x00, 0x00, 0x03, 0x02, 0x01, 0x00, 0xa0,
        0xa1, 0xa2, 0xa3, 0xa4, 0xa5
    ];
    let plaintext: [u8; 23] = [
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e
    ];
    let plaintext_long: [u8; 73] = [
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
        0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
        0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
        0x50
    ];
    let auth_data: [u8; 8] = [
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
    ];
    let expected_ciphertext: [u8; 23] = [
        0x58, 0x8c, 0x97, 0x9a, 0x61, 0xc6, 0x63, 0xd2,
        0xf0, 0x66, 0xd0, 0xc2, 0xc0, 0xf9, 0x89, 0x80,
        0x6d, 0x5f, 0x6b, 0x61, 0xda, 0xc3, 0x84
    ];
    let expected_auth_tag: [u8; 8] = [
        0x17, 0xe8, 0xd1, 0x2c, 0xfd, 0xf9, 0x26, 0xe0
    ];
    let expected_ciphertext_long: [u8; 73] = [
        0x58, 0x8c, 0x97, 0x9a, 0x61, 0xc6, 0x63, 0xd2,
        0xf0, 0x66, 0xd0, 0xc2, 0xc0, 0xf9, 0x89, 0x80,
        0x6d, 0x5f, 0x6b, 0x61, 0xda, 0xc3, 0x84, 0xe0,
        0x44, 0x2d, 0xbe, 0x25, 0xfa, 0x48, 0x2b, 0xa8,
        0x36, 0x0b, 0xbf, 0x01, 0xc0, 0x12, 0x45, 0xa4,
        0x82, 0x9f, 0x20, 0x6c, 0xc3, 0xd6, 0xae, 0x5b,
        0x54, 0x8d, 0xd0, 0xb1, 0x69, 0x2c, 0xec, 0x5e,
        0x95, 0xa5, 0x6b, 0x48, 0xc3, 0xc6, 0xc8, 0x9e,
        0xc7, 0x92, 0x98, 0x9d, 0x26, 0x7d, 0x2a, 0x10,
        0x0b
    ];
    let expected_auth_tag_long: [u8; 8] = [
        0x89, 0xd8, 0xd2, 0x02, 0xc5, 0xcf, 0xae, 0xf4
    ];

    let mut ccm = CCM::new().expect("Failed to create CCM");
    ccm.init(&key).expect("Error with init()");
    let mut auth_tag_out: [u8; 8] = [0; 8];
    let mut cipher_out: [u8; 23] = [0; 23];
    ccm.encrypt(&plaintext, &mut cipher_out,
        &nonce, &auth_data, &mut auth_tag_out).expect("Error with encrypt()");
    assert_eq!(cipher_out, expected_ciphertext);
    assert_eq!(auth_tag_out, expected_auth_tag);
    ccm.init(&key).expect("Error with init()");
    let mut plain_out: [u8; 23] = [0; 23];
    ccm.decrypt(&cipher_out, &mut plain_out,
        &nonce, &auth_data, &auth_tag_out).expect("Error with decrypt()");
    assert_eq!(plain_out, plaintext);

    ccm.init(&key).expect("Error with init()");
    let mut auth_tag_long_out: [u8; 8] = [0; 8];
    let mut cipher_long_out: [u8; 73] = [0; 73];
    ccm.encrypt(&plaintext_long, &mut cipher_long_out,
        &nonce, &auth_data, &mut auth_tag_long_out).expect("Error with encrypt()");
    assert_eq!(cipher_long_out, expected_ciphertext_long);
    assert_eq!(auth_tag_long_out, expected_auth_tag_long);
    ccm.init(&key).expect("Error with init()");
    let mut plain_long_out: [u8; 73] = [0; 73];
    ccm.decrypt(&cipher_long_out, &mut plain_long_out,
        &nonce, &auth_data, &auth_tag_long_out).expect("Error with decrypt()");
    assert_eq!(plain_long_out, plaintext_long);
}

#[test]
#[cfg(aes_ccm)]
fn test_ccm_big_msg() {
    let mut ccm = CCM::new().expect("Failed to create CCM");
    let big_key = b"0123456789abcdeffedcba9876543210";
    let nonce: [u8; 7] = [0x00, 0x00, 0x00, 0x03, 0x02, 0x01, 0x00];
    let auth_data: [u8; 8] = [0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07];
    let mut big_cipher: [u8; 384] = [0; 384];
    let mut big_plain: [u8; 384] = [0; 384];
    let mut auth_tag: [u8; 8] = [0; 8];
    ccm.init(big_key).expect("Error with init()");
    ccm.encrypt(&BIG_MSG, &mut big_cipher,
        &nonce, &auth_data, &mut auth_tag).expect("Error with encrypt()");
    ccm.init(big_key).expect("Error with init()");
    ccm.decrypt(&big_cipher, &mut big_plain,
        &nonce, &auth_data, &auth_tag).expect("Error with decrypt()");
    assert_eq!(big_plain, BIG_MSG);
}

#[test]
#[cfg(aes_cfb)]
fn test_cfb_encrypt_decrypt() {
    let mut cfb = CFB::new().expect("Failed to create CFB");
    let key: [u8; 16] = [
        0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
        0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c
    ];
    let iv: [u8; 16] = [
        0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
        0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f
    ];
    let msg: [u8; 48] = [
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
        0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
        0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
        0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
        0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef
    ];
    let cipher: [u8; 48] = [
        0x3b,0x3f,0xd9,0x2e,0xb7,0x2d,0xad,0x20,
        0x33,0x34,0x49,0xf8,0xe8,0x3c,0xfb,0x4a,
        0xc8,0xa6,0x45,0x37,0xa0,0xb3,0xa9,0x3f,
        0xcd,0xe3,0xcd,0xad,0x9f,0x1c,0xe5,0x8b,
        0x26,0x75,0x1f,0x67,0xa3,0xcb,0xb1,0x40,
        0xb1,0x80,0x8c,0xf1,0x87,0xa4,0xf4,0xdf
    ];
    cfb.init(&key, &iv).expect("Error with init()");
    let mut outbuf: [u8; 48] = [0; 48];
    cfb.encrypt(&msg[0..32], &mut outbuf[0..32]).expect("Error with encrypt()");
    /* Break up encrypt over two operations to test continuation. */
    cfb.encrypt(&msg[32..48], &mut outbuf[32..48]).expect("Error with encrypt()");
    assert_eq!(outbuf, cipher);
    cfb.init(&key, &iv).expect("Error with init()");
    let mut plain: [u8; 48] = [0; 48];
    cfb.decrypt(&outbuf, &mut plain).expect("Error with decrypt()");
    assert_eq!(plain, msg);
}

#[test]
#[cfg(aes_cfb)]
fn test_cfb_big_msg() {
    let mut cfb = CFB::new().expect("Failed to create CFB");
    let big_key = b"0123456789abcdeffedcba9876543210";
    let iv: [u8; 16] = [
        0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
        0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f
    ];
    let mut big_cipher: [u8; 384] = [0; 384];
    let mut big_plain: [u8; 384] = [0; 384];
    cfb.init(big_key, &iv).expect("Error with init()");
    cfb.encrypt(&BIG_MSG, &mut big_cipher).expect("Error with encrypt()");
    cfb.init(big_key, &iv).expect("Error with init()");
    cfb.decrypt(&big_cipher, &mut big_plain).expect("Error with decrypt()");
    assert_eq!(big_plain, BIG_MSG);
}

#[test]
#[cfg(aes_ctr)]
fn test_ctr_encrypt_decrypt() {
    let iv: [u8; 16] = [
        0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
        0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
    ];
    let msg: [u8; 64] = [
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
        0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
        0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
        0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
        0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
        0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
        0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
    ];
    let key: [u8; 16] = [
        0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
        0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c
    ];
    let cipher: [u8; 64] = [
        0x87,0x4d,0x61,0x91,0xb6,0x20,0xe3,0x26,
        0x1b,0xef,0x68,0x64,0x99,0x0d,0xb6,0xce,
        0x98,0x06,0xf6,0x6b,0x79,0x70,0xfd,0xff,
        0x86,0x17,0x18,0x7b,0xb9,0xff,0xfd,0xff,
        0x5a,0xe4,0xdf,0x3e,0xdb,0xd5,0xd3,0x5e,
        0x5b,0x4f,0x09,0x02,0x0d,0xb0,0x3e,0xab,
        0x1e,0x03,0x1d,0xda,0x2f,0xbe,0x03,0xd1,
        0x79,0x21,0x70,0xa0,0xf3,0x00,0x9c,0xee
    ];
    let mut ctr = CTR::new().expect("Failed to create CTR");
    ctr.init(&key, &iv).expect("Error with init()");
    let mut outbuf: [u8; 64] = [0; 64];
    ctr.encrypt(&msg, &mut outbuf).expect("Error with encrypt()");
    assert_eq!(outbuf, cipher);
    ctr.init(&key, &iv).expect("Error with init()");
    let mut plain: [u8; 64] = [0; 64];
    ctr.decrypt(&outbuf, &mut plain).expect("Error with decrypt()");
    assert_eq!(plain, msg);
}

#[test]
#[cfg(aes_ctr)]
fn test_ctr_big_msg() {
    let mut ctr = CTR::new().expect("Failed to create CTR");
    let big_key = b"0123456789abcdeffedcba9876543210";
    let iv: [u8; 16] = [
        0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
        0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f
    ];
    let mut big_cipher: [u8; 384] = [0; 384];
    let mut big_plain: [u8; 384] = [0; 384];
    ctr.init(big_key, &iv).expect("Error with init()");
    ctr.encrypt(&BIG_MSG, &mut big_cipher).expect("Error with encrypt()");
    ctr.init(big_key, &iv).expect("Error with init()");
    ctr.decrypt(&big_cipher, &mut big_plain).expect("Error with decrypt()");
    assert_eq!(big_plain, BIG_MSG);
}

#[test]
#[cfg(aes_eax)]
fn test_eax_one_shot_encrypt_decrypt() {
    let key: [u8; 16] = [
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
    ];
    let nonce: [u8; 16] = [
        0x3c, 0x8c, 0xc2, 0x97, 0x0a, 0x00, 0x8f, 0x75,
        0xcc, 0x5b, 0xea, 0xe2, 0x84, 0x72, 0x58, 0xc2
    ];
    let auth: &[u8] = &[];
    let msg: [u8; 32] = [
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
        0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11
    ];
    let expected_cipher: [u8; 32] = [
        0x3c, 0x44, 0x1f, 0x32, 0xce, 0x07, 0x82, 0x23,
        0x64, 0xd7, 0xa2, 0x99, 0x0e, 0x50, 0xbb, 0x13,
        0xd7, 0xb0, 0x2a, 0x26, 0x96, 0x9e, 0x4a, 0x93,
        0x7e, 0x5e, 0x90, 0x73, 0xb0, 0xd9, 0xc9, 0x68
    ];
    let expected_auth_tag: [u8; 16] = [
        0xdb, 0x90, 0xbd, 0xb3, 0xda, 0x3d, 0x00, 0xaf,
        0xd0, 0xfc, 0x6a, 0x83, 0x55, 0x1d, 0xa9, 0x5e
    ];
    let mut cipher: [u8; 32] = [0; 32];
    let mut auth_tag: [u8; 16] = [0; 16];
    EAX::encrypt(&msg, &mut cipher, &key, &nonce, auth, &mut auth_tag).expect("Error with encrypt()");
    assert_eq!(cipher, expected_cipher);
    assert_eq!(auth_tag, expected_auth_tag);
    let mut plain: [u8; 32] = [0; 32];
    EAX::decrypt(&cipher, &mut plain, &key, &nonce, auth, &auth_tag).expect("Error with decrypt()");
    assert_eq!(plain, msg);
}

#[test]
#[cfg(aes_ecb)]
fn test_ecb_encrypt_decrypt() {
    let mut ecb = ECB::new().expect("Failed to create ECB");
    let key_128: &[u8; 16] = b"0123456789abcdef";
    let msg: [u8; 16] = [
        0x6e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74,
        0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20
    ];
    let verify_ecb_128: [u8; 16] = [
        0xd0, 0xc9, 0xd9, 0xc9, 0x40, 0xe8, 0x97, 0xb6,
        0xc8, 0x8c, 0x33, 0x3b, 0xb5, 0x8f, 0x85, 0xd1
    ];
    ecb.init_encrypt(key_128).expect("Error with init_encrypt()");
    let mut outbuf: [u8; 16] = [0; 16];
    ecb.encrypt(&msg, &mut outbuf).expect("Error with encrypt()");
    assert_eq!(&outbuf, &verify_ecb_128);
    outbuf = [0; 16];
    ecb.init_decrypt(key_128).expect("Error with init_decrypt()");
    ecb.decrypt(&verify_ecb_128, &mut outbuf).expect("Error with decrypt()");
    assert_eq!(&outbuf, &msg);
}

#[test]
#[cfg(aes_gcm)]
fn test_gcm_encrypt_decrypt() {
    let key: [u8; 16] = [
        0x29, 0x8e, 0xfa, 0x1c, 0xcf, 0x29, 0xcf, 0x62,
        0xae, 0x68, 0x24, 0xbf, 0xc1, 0x95, 0x57, 0xfc
    ];
    let iv: [u8; 12] = [
        0x6f, 0x58, 0xa9, 0x3f, 0xe1, 0xd2, 0x07, 0xfa,
        0xe4, 0xed, 0x2f, 0x6d
    ];
    let plain: [u8; 32] = [
        0xcc, 0x38, 0xbc, 0xcd, 0x6b, 0xc5, 0x36, 0xad,
        0x91, 0x9b, 0x13, 0x95, 0xf5, 0xd6, 0x38, 0x01,
        0xf9, 0x9f, 0x80, 0x68, 0xd6, 0x5c, 0xa5, 0xac,
        0x63, 0x87, 0x2d, 0xaf, 0x16, 0xb9, 0x39, 0x01
    ];
    let auth: [u8; 16] = [
        0x02, 0x1f, 0xaf, 0xd2, 0x38, 0x46, 0x39, 0x73,
        0xff, 0xe8, 0x02, 0x56, 0xe5, 0xb1, 0xc6, 0xb1
    ];
    let expected_cipher: [u8; 32] = [
        0xdf, 0xce, 0x4e, 0x9c, 0xd2, 0x91, 0x10, 0x3d,
        0x7f, 0xe4, 0xe6, 0x33, 0x51, 0xd9, 0xe7, 0x9d,
        0x3d, 0xfd, 0x39, 0x1e, 0x32, 0x67, 0x10, 0x46,
        0x58, 0x21, 0x2d, 0xa9, 0x65, 0x21, 0xb7, 0xdb
    ];
    let expected_auth_tag: [u8; 16] = [
        0x54, 0x24, 0x65, 0xef, 0x59, 0x93, 0x16, 0xf7,
        0x3a, 0x7a, 0x56, 0x05, 0x09, 0xa2, 0xd9, 0xf2
    ];
    let mut gcm = GCM::new().expect("Failed to create GCM");
    gcm.init(&key).expect("Error with init()");
    let mut cipher: [u8; 32] = [0; 32];
    let mut auth_tag: [u8; 16] = [0; 16];
    gcm.encrypt(&plain, &mut cipher, &iv, &auth, &mut auth_tag).expect("Error with encrypt()");
    assert_eq!(cipher, expected_cipher);
    assert_eq!(auth_tag, expected_auth_tag);
    let mut plain_out: [u8; 32] = [0; 32];
    gcm.decrypt(&cipher, &mut plain_out, &iv, &auth, &auth_tag).expect("Error with decrypt()");
    assert_eq!(plain_out, plain);
}

#[test]
#[cfg(aes_gcm_stream)]
fn test_gcmstream_encrypt_decrypt() {
    let plain: [u8; 60] = [
        0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
        0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
        0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
        0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
        0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
        0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
        0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
        0xba, 0x63, 0x7b, 0x39
    ];
    let auth: [u8; 20] = [
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xab, 0xad, 0xda, 0xd2
    ];
    let key: [u8; 32] = [
        0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
        0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08,
        0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
        0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08
    ];
    let iv: [u8; 12] = [
        0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
        0xde, 0xca, 0xf8, 0x88
    ];
    let expected_cipher: [u8; 60] = [
        0x52, 0x2d, 0xc1, 0xf0, 0x99, 0x56, 0x7d, 0x07,
        0xf4, 0x7f, 0x37, 0xa3, 0x2a, 0x84, 0x42, 0x7d,
        0x64, 0x3a, 0x8c, 0xdc, 0xbf, 0xe5, 0xc0, 0xc9,
        0x75, 0x98, 0xa2, 0xbd, 0x25, 0x55, 0xd1, 0xaa,
        0x8c, 0xb0, 0x8e, 0x48, 0x59, 0x0d, 0xbb, 0x3d,
        0xa7, 0xb0, 0x8b, 0x10, 0x56, 0x82, 0x88, 0x38,
        0xc5, 0xf6, 0x1e, 0x63, 0x93, 0xba, 0x7a, 0x0a,
        0xbc, 0xc9, 0xf6, 0x62
    ];
    let expected_auth_tag: [u8; 16] = [
        0x76, 0xfc, 0x6e, 0xce, 0x0f, 0x4e, 0x17, 0x68,
        0xcd, 0xdf, 0x88, 0x53, 0xbb, 0x2d, 0x55, 0x1b
    ];
    let mut gcmstream = GCMStream::new().expect("Failed to create GCMStream");
    for chunk_size in 1..=auth.len() {
        gcmstream.init(&key, &iv).expect("Error with init()");
        let mut cipher: [u8; 60] = [0; 60];
        let mut i = 0;
        while i < auth.len() {
            let mut end = i + chunk_size;
            if end > auth.len() {
                end = auth.len()
            }
            gcmstream.encrypt_update(&plain[0..0], &mut cipher[0..0], &auth[i..end]).expect("Error with encrypt_update()");
            i += chunk_size;
        }
        i = 0;
        while i < plain.len() {
            let mut end = i + chunk_size;
            if end > plain.len() {
                end = plain.len()
            }
            gcmstream.encrypt_update(&plain[i..end], &mut cipher[i..end], &auth[0..0]).expect("Error with encrypt_update()");
            i += chunk_size;
        }
        let mut auth_tag: [u8; 16] = [0; 16];
        gcmstream.encrypt_final(&mut auth_tag).expect("Error with encrypt_final()");
        assert_eq!(cipher, expected_cipher);
        assert_eq!(auth_tag, expected_auth_tag);
    }
}

#[test]
#[cfg(aes_ofb)]
fn test_ofb_encrypt_decrypt() {
    let key: [u8; 32] = [
        0xc4,0xc7,0xfa,0xd6,0x53,0x5c,0xb8,0x71,
        0x4a,0x5c,0x40,0x77,0x9a,0x8b,0xa1,0xd2,
        0x53,0x3e,0x23,0xb4,0xb2,0x58,0x73,0x2a,
        0x5b,0x78,0x01,0xf4,0xe3,0x71,0xa7,0x94
    ];
    let iv: [u8; 16] = [
        0x5e,0xb9,0x33,0x13,0xb8,0x71,0xff,0x16,
        0xb9,0x8a,0x9b,0xcb,0x43,0x33,0x0d,0x6f
    ];
    let plain: [u8; 48] = [
        0x6d,0x0b,0xb0,0x79,0x63,0x84,0x71,0xe9,
        0x39,0xd4,0x53,0x14,0x86,0xc1,0x4c,0x25,
        0x9a,0xee,0xc6,0xf3,0xc0,0x0d,0xfd,0xd6,
        0xc0,0x50,0xa8,0xba,0xa8,0x20,0xdb,0x71,
        0xcc,0x12,0x2c,0x4e,0x0c,0x17,0x15,0xef,
        0x55,0xf3,0x99,0x5a,0x6b,0xf0,0x2a,0x4c
    ];
    let expected_cipher: [u8; 48] = [
        0x0f,0x54,0x61,0x71,0x59,0xd0,0x3f,0xfc,
        0x1b,0xfa,0xfb,0x60,0x29,0x30,0xd7,0x00,
        0xf4,0xa4,0xa8,0xe6,0xdd,0x93,0x94,0x46,
        0x64,0xd2,0x19,0xc4,0xc5,0x4d,0xde,0x1b,
        0x04,0x53,0xe1,0x73,0xf5,0x18,0x74,0xae,
        0xfd,0x64,0xa2,0xe1,0xe2,0x76,0x13,0xb0
    ];
    let mut ofb = OFB::new().expect("Failed to create OFB");
    ofb.init(&key, &iv).expect("Error with init()");
    let mut cipher: [u8; 48] = [0; 48];
    ofb.encrypt(&plain, &mut cipher).expect("Error with encrypt()");
    assert_eq!(cipher, expected_cipher);
    ofb.init(&key, &iv).expect("Error with init()");
    let mut plain_out: [u8; 48] = [0; 48];
    ofb.decrypt(&cipher, &mut plain_out).expect("Error with decrypt()");
    assert_eq!(plain_out, plain);
}

#[test]
#[cfg(aes_xts)]
fn test_xts_one_shot() {
    let key: [u8; 32] = [
        0xa1, 0xb9, 0x0c, 0xba, 0x3f, 0x06, 0xac, 0x35,
        0x3b, 0x2c, 0x34, 0x38, 0x76, 0x08, 0x17, 0x62,
        0x09, 0x09, 0x23, 0x02, 0x6e, 0x91, 0x77, 0x18,
        0x15, 0xf2, 0x9d, 0xab, 0x01, 0x93, 0x2f, 0x2f
    ];
    let tweak: [u8; 16] = [
        0x4f, 0xae, 0xf7, 0x11, 0x7c, 0xda, 0x59, 0xc6,
        0x6e, 0x4b, 0x92, 0x01, 0x3e, 0x76, 0x8a, 0xd5
    ];
    let plain: [u8; 16] = [
        0xeb, 0xab, 0xce, 0x95, 0xb1, 0x4d, 0x3c, 0x8d,
        0x6f, 0xb3, 0x50, 0x39, 0x07, 0x90, 0x31, 0x1c
    ];
    let expected_cipher: [u8; 16] = [
        0x77, 0x8a, 0xe8, 0xb4, 0x3c, 0xb9, 0x8d, 0x5a,
        0x82, 0x50, 0x81, 0xd5, 0xbe, 0x47, 0x1c, 0x63
    ];
    let partial: [u8; 24] = [
        0xeb, 0xab, 0xce, 0x95, 0xb1, 0x4d, 0x3c, 0x8d,
        0x6f, 0xb3, 0x50, 0x39, 0x07, 0x90, 0x31, 0x1c,
        0x6e, 0x4b, 0x92, 0x01, 0x3e, 0x76, 0x8a, 0xd5
    ];
    let expected_partial_cipher: [u8; 24] = [
        0x2b, 0xf7, 0x2c, 0xf3, 0xeb, 0x85, 0xef, 0x7b,
        0x0b, 0x76, 0xa0, 0xaa, 0xf3, 0x3f, 0x25, 0x8b,
        0x77, 0x8a, 0xe8, 0xb4, 0x3c, 0xb9, 0x8d, 0x5a
    ];

    let mut xts = XTS::new().expect("Failed to create XTS");
    xts.init_encrypt(&key).expect("Error with init_encrypt()");
    let mut cipher: [u8; 16] = [0; 16];
    xts.encrypt(&plain, &mut cipher, &tweak).expect("Error with encrypt()");
    assert_eq!(cipher, expected_cipher);
    xts.init_decrypt(&key).expect("Error with init_decrypt()");
    let mut plain_out: [u8; 16] = [0; 16];
    xts.decrypt(&cipher, &mut plain_out, &tweak).expect("Error with decrypt()");
    assert_eq!(plain_out, plain);

    xts.init_encrypt(&key).expect("Error with init_encrypt()");
    let mut partial_cipher: [u8; 24] = [0; 24];
    xts.encrypt(&partial, &mut partial_cipher, &tweak).expect("Error with encrypt()");
    assert_eq!(partial_cipher, expected_partial_cipher);
    xts.init_decrypt(&key).expect("Error with init_decrypt()");
    let mut partial_out: [u8; 24] = [0; 24];
    xts.decrypt(&partial_cipher, &mut partial_out, &tweak).expect("Error with decrypt()");
    assert_eq!(partial_out, partial);
}

#[test]
#[cfg(aes_xts)]
fn test_xts_sector_128() {
    let keys: [u8; 32] = [
        0xa3, 0xe4, 0x0d, 0x5b, 0xd4, 0xb6, 0xbb, 0xed,
        0xb2, 0xd1, 0x8c, 0x70, 0x0a, 0xd2, 0xdb, 0x22,
        0x10, 0xc8, 0x11, 0x90, 0x64, 0x6d, 0x67, 0x3c,
        0xbc, 0xa5, 0x3f, 0x13, 0x3e, 0xab, 0x37, 0x3c
    ];
    let plain: [u8; 16] =[
        0x20, 0xe0, 0x71, 0x94, 0x05, 0x99, 0x3f, 0x09,
        0xa6, 0x6a, 0xe5, 0xbb, 0x50, 0x0e, 0x56, 0x2c
    ];
    let expected_cipher: [u8; 16] = [
        0x74, 0x62, 0x35, 0x51, 0x21, 0x02, 0x16, 0xac,
        0x92, 0x6b, 0x96, 0x50, 0xb6, 0xd3, 0xfa, 0x52
    ];
    let sector = 141;

    let mut xts = XTS::new().expect("Failed to create XTS");
    xts.init_encrypt(&keys).expect("Error with init_encrypt()");
    let mut cipher: [u8; 16] = [0; 16];
    xts.encrypt_sector(&plain, &mut cipher, sector).expect("Error with encrypt_sector()");
    assert_eq!(cipher, expected_cipher);

    xts.init_decrypt(&keys).expect("Error with init_decrypt()");
    let mut plain_out: [u8; 16] = [0; 16];
    xts.decrypt_sector(&cipher, &mut plain_out, sector).expect("Error with decrypt_sector()");
    assert_eq!(plain_out, plain);
}

#[test]
#[cfg(aes_xts)]
fn test_xts_sector_256() {
    let keys: [u8; 64] = [
        0xef, 0x01, 0x0c, 0xa1, 0xa3, 0x66, 0x3e, 0x32,
        0x53, 0x43, 0x49, 0xbc, 0x0b, 0xae, 0x62, 0x23,
        0x2a, 0x15, 0x73, 0x34, 0x85, 0x68, 0xfb, 0x9e,
        0xf4, 0x17, 0x68, 0xa7, 0x67, 0x4f, 0x50, 0x7a,
        0x72, 0x7f, 0x98, 0x75, 0x53, 0x97, 0xd0, 0xe0,
        0xaa, 0x32, 0xf8, 0x30, 0x33, 0x8c, 0xc7, 0xa9,
        0x26, 0xc7, 0x73, 0xf0, 0x9e, 0x57, 0xb3, 0x57,
        0xcd, 0x15, 0x6a, 0xfb, 0xca, 0x46, 0xe1, 0xa0
    ];
    let plain: [u8; 32] =[
        0xed, 0x98, 0xe0, 0x17, 0x70, 0xa8, 0x53, 0xb4,
        0x9d, 0xb9, 0xe6, 0xaa, 0xf8, 0x8f, 0x0a, 0x41,
        0xb9, 0xb5, 0x6e, 0x91, 0xa5, 0xa2, 0xb1, 0x1d,
        0x40, 0x52, 0x92, 0x54, 0xf5, 0x52, 0x3e, 0x75
    ];
    let expected_cipher: [u8; 32] = [
        0xca, 0x20, 0xc5, 0x5e, 0x8d, 0xc1, 0x49, 0x68,
        0x7d, 0x25, 0x41, 0xde, 0x39, 0xc3, 0xdf, 0x63,
        0x00, 0xbb, 0x5a, 0x16, 0x3c, 0x10, 0xce, 0xd3,
        0x66, 0x6b, 0x13, 0x57, 0xdb, 0x8b, 0xd3, 0x9d
    ];
    let sector = 187;

    let mut xts = XTS::new().expect("Failed to create XTS");
    xts.init_encrypt(&keys).expect("Error with init_encrypt()");
    let mut cipher: [u8; 32] = [0; 32];
    xts.encrypt_sector(&plain, &mut cipher, sector).expect("Error with encrypt_sector()");
    assert_eq!(cipher, expected_cipher);

    xts.init_decrypt(&keys).expect("Error with init_decrypt()");
    let mut plain_out: [u8; 32] = [0; 32];
    xts.decrypt_sector(&cipher, &mut plain_out, sector).expect("Error with decrypt_sector()");
    assert_eq!(plain_out, plain);
}

#[test]
#[cfg(aes_xts)]
fn test_xts_consecutive_sectors() {
    let keys: [u8; 32] = [
        0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
        0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
        0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
        0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22
    ];
    let plain: [u8; 544] =[
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
        0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
        0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
        0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
        0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
        0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
        0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
        0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
        0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
        0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
        0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
        0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
        0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
        0xfc, 0xfd, 0xfe, 0xff,

        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
        0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
        0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
        0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
        0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
        0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
        0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
        0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
        0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
        0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
        0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
        0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
        0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
        0xfc, 0xfd, 0xfe, 0xff,

        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f
    ];
    let expected_cipher: [u8; 544] = [
        0xb9, 0x6b, 0x2b, 0xfd, 0x61, 0x87, 0x84, 0xd5, 0x26, 0xd2, 0x8c, 0x62,
        0x63, 0x01, 0xca, 0x46, 0xb1, 0x82, 0xfa, 0xdc, 0xbc, 0x32, 0x18, 0xe9,
        0xda, 0xe6, 0xda, 0xd1, 0x1a, 0x52, 0x77, 0xca, 0xdb, 0x0e, 0xbe, 0x37,
        0x88, 0x36, 0x1c, 0x87, 0x16, 0x60, 0xfe, 0xa8, 0x9e, 0xf6, 0x48, 0x64,
        0x94, 0x34, 0x64, 0xed, 0xf6, 0x9a, 0xc5, 0x28, 0xc9, 0xed, 0x64, 0x80,
        0x85, 0xd8, 0x93, 0xa7, 0x50, 0xb1, 0x9d, 0x2f, 0x1e, 0x34, 0xcc, 0xb4,
        0x03, 0xfb, 0x6b, 0x43, 0x21, 0xa8, 0x5b, 0xc6, 0x59, 0x13, 0xd2, 0xb5,
        0xf5, 0x7b, 0xf6, 0xb2, 0xa4, 0x7a, 0xd2, 0x50, 0x26, 0xcb, 0xa4, 0x83,
        0xc3, 0x56, 0xb0, 0xb1, 0x14, 0x34, 0x12, 0x1b, 0xea, 0x26, 0x97, 0x24,
        0x54, 0xcc, 0x32, 0x4c, 0xa4, 0xc2, 0xa3, 0x07, 0xfa, 0x30, 0xa9, 0xf0,
        0x91, 0x17, 0x60, 0x68, 0x88, 0x7f, 0x34, 0x7e, 0xbd, 0x20, 0x33, 0x95,
        0x6e, 0xc0, 0xb6, 0x2b, 0xff, 0x7e, 0x61, 0x35, 0x9a, 0x88, 0xff, 0xd9,
        0x69, 0x21, 0xe7, 0x8f, 0x45, 0x02, 0xf9, 0xd7, 0xeb, 0xa6, 0x53, 0xf1,
        0x73, 0x04, 0xf1, 0x0b, 0x85, 0xc6, 0x1f, 0x4a, 0x51, 0x2f, 0x95, 0x87,
        0x5a, 0x67, 0x37, 0xb2, 0x87, 0xf7, 0xbe, 0x2a, 0x17, 0x57, 0xca, 0xfc,
        0xdd, 0x5f, 0x37, 0x48, 0x78, 0xbd, 0xfa, 0x75, 0xc9, 0xfa, 0x86, 0x7e,
        0xc4, 0x0f, 0x60, 0x85, 0xce, 0x12, 0x44, 0x7c, 0xd9, 0xb2, 0x50, 0xd9,
        0x57, 0x85, 0xa5, 0xd7, 0x68, 0x59, 0x03, 0x09, 0x97, 0x2e, 0x8e, 0xa5,
        0xe3, 0x98, 0xac, 0x16, 0xfb, 0x6d, 0x54, 0xc5, 0x5d, 0x7a, 0x33, 0x44,
        0x0a, 0x39, 0x91, 0xcc, 0x9f, 0x67, 0xf9, 0x89, 0xbb, 0x62, 0x02, 0xc4,
        0x22, 0xec, 0xcf, 0x97, 0x69, 0x81, 0x3d, 0x00, 0xfd, 0xeb, 0x55, 0x08,
        0xa2, 0xff, 0x97, 0xaa, 0x79, 0xde, 0x3c, 0x8a, 0x78, 0x71, 0x73, 0xa2,
        0x98, 0x2f, 0xd8, 0x5c, 0x62, 0x1c, 0x5c, 0x23, 0x0a, 0xd1, 0xf1, 0x81,
        0x8a, 0x12, 0xe7, 0x4d, 0xdd, 0x4f, 0xd4, 0xf1, 0xe8, 0x0f, 0x25, 0x79,
        0x45, 0x4a, 0x49, 0x49, 0x7e, 0x56, 0x91, 0x4e, 0xaa, 0xba, 0x18, 0xe1,
        0xe4, 0xbe, 0x21, 0xdc, 0x58, 0x60, 0x6f, 0x6a, 0x7f, 0xdc, 0x5e, 0x74,
        0x47, 0xbf, 0xeb, 0x84, 0xc4, 0x1e, 0x5a, 0x61, 0x64, 0xc8, 0x63, 0x68,
        0xfa, 0x17, 0x9c, 0xac, 0x60, 0x1c, 0xa5, 0x6e, 0x00, 0x21, 0x93, 0x3c,
        0xd7, 0xbb, 0x73, 0x45, 0xf7, 0x34, 0x81, 0x6c, 0xfa, 0xf2, 0x33, 0xfd,
        0xb1, 0x40, 0x30, 0x6b, 0x30, 0xd1, 0x83, 0x5e, 0x2e, 0x7a, 0xce, 0xa6,
        0x12, 0x2a, 0x15, 0x03, 0x78, 0x29, 0xb9, 0x07, 0xae, 0xe7, 0xc2, 0x78,
        0x74, 0x72, 0xa5, 0x0e, 0x6b, 0x1f, 0x78, 0xf2, 0x5a, 0x69, 0xb6, 0x2b,
        0x99, 0x94, 0x1f, 0x89, 0xd1, 0x21, 0x14, 0x4a, 0x54, 0xab, 0x5a, 0x9f,
        0xaa, 0xa7, 0x96, 0x0a, 0x21, 0xce, 0x30, 0xb6, 0x70, 0x81, 0xe9, 0xd3,
        0x71, 0xc0, 0xf1, 0x15, 0xe2, 0xf6, 0xd3, 0xcc, 0x41, 0x15, 0x9d, 0xd5,
        0xa3, 0xa4, 0xe0, 0xf8, 0x62, 0xc4, 0x76, 0x65, 0x63, 0x89, 0xa7, 0xe2,
        0xfb, 0xf5, 0xc9, 0x80, 0x15, 0x5b, 0xc1, 0x59, 0xb2, 0xd0, 0x01, 0x3a,
        0xf9, 0xab, 0x5b, 0x79, 0x54, 0xed, 0x6b, 0xf9, 0x1d, 0x9d, 0x87, 0x63,
        0x80, 0x4f, 0xec, 0x9c, 0x4f, 0xad, 0x97, 0x04, 0xff, 0x62, 0x4a, 0x17,
        0xc0, 0x09, 0x2a, 0x2c, 0x23, 0x4b, 0xc3, 0xb6, 0x6d, 0xed, 0xdb, 0x1a,
        0x6f, 0x56, 0x2b, 0x78, 0x92, 0x3a, 0x5c, 0x7f, 0xb2, 0x63, 0xd3, 0xd5,
        0x1a, 0xbe, 0xc2, 0x34, 0xc8, 0xad, 0x36, 0xb7, 0x12, 0xb8, 0xe1, 0xb7,
        0x52, 0x7f, 0x16, 0x84, 0x2c, 0x47, 0x7e, 0xf2, 0xa5, 0x36, 0x2e, 0xad,
        0xe7, 0xbb, 0xc0, 0x6f, 0x27, 0x8e, 0x41, 0x08, 0x75, 0xe5, 0xff, 0xde,
        0x08, 0x9f, 0x8c, 0x91, 0xba, 0xc9, 0x9d, 0x9f, 0x27, 0x90, 0x50, 0x44,
        0x24, 0xe7, 0x3d, 0x6f
    ];
    let sector = 0x000000ffffffffff;
    let sector_size = 512;

    let mut xts = XTS::new().expect("Failed to create XTS");
    xts.init_encrypt(&keys).expect("Error with init_encrypt()");
    let mut cipher: [u8; 544] = [0; 544];
    xts.encrypt_consecutive_sectors(&plain, &mut cipher, sector, sector_size).expect("Error with encrypt_consecutive_sectors()");
    assert_eq!(cipher, expected_cipher);

    xts.init_decrypt(&keys).expect("Error with init_decrypt()");
    let mut plain_out: [u8; 544] = [0; 544];
    xts.decrypt_consecutive_sectors(&cipher, &mut plain_out, sector, sector_size).expect("Error with decrypt_consecutive_sectors()");
    assert_eq!(plain_out, plain);
}

#[test]
#[cfg(aes_xts_stream)]
fn test_xtsstream() {
    let keys: [u8; 32] = [
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    ];
    let tweak: [u8; 16] = [
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    ];
    let plain: [u8; 40] = [
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0xff, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20
    ];
    let expected_cipher: [u8; 40] = [
        0xA2, 0x07, 0x47, 0x76, 0x3F, 0xEC, 0x0C, 0x23,
        0x1B, 0xD0, 0xBD, 0x46, 0x9A, 0x27, 0x38, 0x12,
        0x95, 0x02, 0x3D, 0x5D, 0xC6, 0x94, 0x51, 0x36,
        0xA0, 0x85, 0xD2, 0x69, 0x6E, 0x87, 0x0A, 0xBF,
        0xB5, 0x5A, 0xDD, 0xCB, 0x80, 0xE0, 0xFC, 0xCD
    ];

    let mut xtsstream = XTSStream::new().expect("Failed to create XTSStream");
    xtsstream.init_encrypt(&keys, &tweak).expect("Error with init_encrypt()");
    let mut cipher: [u8; 40] = [0; 40];
    xtsstream.encrypt_update(&plain[0..16], &mut cipher[0..16]).expect("Error with encrypt_update()");
    xtsstream.encrypt_final(&plain[16..40], &mut cipher[16..40]).expect("Error with encrypt_final()");
    assert_eq!(cipher, expected_cipher);

    xtsstream.init_decrypt(&keys, &tweak).expect("Error with init_decrypt()");
    let mut plain_out: [u8; 40] = [0; 40];
    xtsstream.decrypt_update(&cipher[0..16], &mut plain_out[0..16]).expect("Error with decrypt_update()");
    xtsstream.decrypt_final(&cipher[16..40], &mut plain_out[16..40]).expect("Error with decrypt_final()");
    assert_eq!(plain_out, plain);
}

#[test]
#[cfg(aes_xts_stream)]
fn test_xtsstream_big_msg() {
    let key: [u8; 32] = [
        0xa1, 0xb9, 0x0c, 0xba, 0x3f, 0x06, 0xac, 0x35,
        0x3b, 0x2c, 0x34, 0x38, 0x76, 0x08, 0x17, 0x62,
        0x09, 0x09, 0x23, 0x02, 0x6e, 0x91, 0x77, 0x18,
        0x15, 0xf2, 0x9d, 0xab, 0x01, 0x93, 0x2f, 0x2f
    ];
    let tweak: [u8; 16] = [
        0x4f, 0xae, 0xf7, 0x11, 0x7c, 0xda, 0x59, 0xc6,
        0x6e, 0x4b, 0x92, 0x01, 0x3e, 0x76, 0x8a, 0xd5
    ];

    let mut xtsstream = XTSStream::new().expect("Failed to create XTSStream");
    xtsstream.init_encrypt(&key, &tweak).expect("Error with init_encrypt()");
    let mut cipher: [u8; 384] = [0; 384];
    let mut i = 0;
    while i < BIG_MSG.len() {
        let remaining = BIG_MSG.len() - i;
        if remaining < 32 {
            xtsstream.encrypt_final(&BIG_MSG[i..BIG_MSG.len()], &mut cipher[i..BIG_MSG.len()]).expect("Error with encrypt_final()");
            i += remaining;
        } else {
            let end = i + 16;
            xtsstream.encrypt_update(&BIG_MSG[i..end], &mut cipher[i..end]).expect("Error with encrypt_update()");
            i += 16;
        }
    }

    xtsstream.init_decrypt(&key, &tweak).expect("Error with init_decrypt()");
    let mut plain_out: [u8; 384] = [0; 384];
    let mut i = 0;
    while i < BIG_MSG.len() {
        let remaining = BIG_MSG.len() - i;
        if remaining < 32 {
            xtsstream.decrypt_final(&cipher[i..BIG_MSG.len()], &mut plain_out[i..BIG_MSG.len()]).expect("Error with decrypt_final()");
            i += remaining;
        } else {
            let end = i + 16;
            xtsstream.decrypt_update(&cipher[i..end], &mut plain_out[i..end]).expect("Error with decrypt_update()");
            i += 16;
        }
    }

    assert_eq!(plain_out, BIG_MSG);
}
