# -*- coding: binary -*-

# Module to get version of splunk app
module Msf::Exploit::Remote::HTTP::Splunk::Version
  # Extracts the Splunk version information using authenticated cookie if available
  #
  # @param cookie_string [String] Valid cookie if available
  # @return [String, nil] Splunk version if found, nil otherwise
  def splunk_version(cookie_string = nil)
    version = splunk_version_authenticated(cookie_string) if !cookie_string.nil?
    return version if version

    version = splunk_login_version
    return version if version

    nil
  end

  private

  # Extracts splunk version from splunk user page using valid cookie
  #
  # @param cookie_string [String] Valid cookie
  # @return [String] Splunk version
  def splunk_version_authenticated(cookie_string)
    res = send_request_cgi({
      'uri' => splunk_user_page,
      'vars_get' => {
        'output_mode' => 'json'
      },
      'headers' => {
        'Cookie' => cookie_string
      }
    })

    return nil unless res&.code == 200

    body = res.get_json_document
    body.dig('generator', 'version')
  end

  # Tries to extract splunk verion from login page
  #
  # @return [String, nil] Splunk version if found, otherwise nil
  def splunk_login_version
    res = send_request_cgi({
      'uri' => splunk_url_login,
      'method' => 'GET'
    })

    if res
      match = res.body.match(/Splunk \d+\.\d+\.\d+/)
      return match[0].split[1] if match
    end
  end

  # Tries to extract splunk verion from home page
  #
  # @param cookie_string [String] Valid cookie
  # @return [Rex::Version, nil] Splunk version if found, otherwise nil
  def splunk_home_version(cookie_string = nil)
    res = send_request_cgi(
      'uri' => splunk_home,
      'method' => 'GET',
      'cookie' => cookie_string
    )
    unless res&.code == 200
      fail_with(Msf::Module::Failure::UnexpectedReply, "#{peer} Server did not respond with the expected HTTP 200")
    end

    script_tag = res.get_html_document.at('script:contains("__splunkd_partials__")')
    return nil unless script_tag

    script_text = script_tag.text
    return nil unless script_text

    # search json string like {}
    json_str = script_text[/\{.*\}/m]
    return nil unless json_str

    begin
      splunkd_partials = JSON.parse(json_str)
    rescue JSON::ParserError
      return nil
    end

    version = splunkd_partials.dig('/services/server/info', 'entry', 0, 'content', 'version')
    return nil unless version

    Rex::Version.new(version)
  end
end
